//------------------------------------------------------------
// Генератор класса для вещественного вектора фиксированной размерности.
//------------------------------------------------------------
// Включение этого файла должно ОБЯЗАТЕЛЬНО предваряться описанием
//   #define RDSVECTOR имя_класса
// При этом генерируется класс с именем "имя_класса".
// Обязательно должно быть описание
//   #define RDSVECTORSIZE размер
// для размера вектора.
// При наличии описания
//   #define RDSVECTORALIASES RDSVECTOR_ALIAS(x,0) RDSVECTOR_ALIAS(y,1) ...
// будут добавлены соответствующие псевдонимы.
//------------------------------------------------------------

#ifdef RDSVECTOR

#ifndef RDSVECTORSIZE
  #error RDSVECTORSIZE must be defined
#else
  #if RDSVECTORSIZE<=0
    #error Bad RDSVECTORSIZE
  #endif
#endif

#include <math.h>
#include <string.h> // Для memcpy

class RDSVECTOR
{ public:
    double _Value[RDSVECTORSIZE];
  
    inline unsigned int Size(void)const{return RDSVECTORSIZE; };
    inline void SetValueNoCheck(unsigned int i,double v){_Value[i]=v;};
    inline double GetValueNoCheck(unsigned int i)const{return _Value[i];};

    #ifdef RDSVECTORALIASES
      #ifdef RDSVECTOR_ALIAS
        #undef RDSVECTOR_ALIAS
      #endif
      // Псевдонимы для удобства
      // RDSVECTOR_ALIAS(имя,индекс)
      // Теперь:
      //   x=имя()   - возврат _Value[индекс]
      //   _имя()=x  - возврат _Value[индекс] как lvalue
      //   имя(x)    - присваивание _Value[индекс]
      #define RDSVECTOR_ALIAS(var,index) inline double &_##var(void){return _Value[index];}; \
         inline double var(void)const{return _Value[index];}; \
         inline double var(double __value){return (_Value[index]=__value);};
      RDSVECTORALIASES
      #undef RDSVECTOR_ALIAS
      // Функция проверки индексов у псевдонимов
      static BOOL CheckAliases(void)
        { int maxindex=-1;
          #define RDSVECTOR_ALIAS(var,index) if(index>maxindex) maxindex=index;
            RDSVECTORALIASES
          #undef RDSVECTOR_ALIAS
          return maxindex<RDSVECTORSIZE;
        };
      #undef RDSVECTORALIASES
    #endif

    // Максимальное абсолютное значение
    double MaxAbs(void)const 
      { double m=fabs(_Value[0]);
        #if RDSVECTORSIZE>1
          for(unsigned int i=1;i<RDSVECTORSIZE;i++)
            { double _m=fabs(_Value[i]); if(m<_m) m=_m; }
        #endif
        return m;
      };
      
    // Занести константу
    void SetConst(double c)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]=c;
      };
      
    // Численная производная по двум точкам
    void CalcDeriv(const RDSVECTOR s_t,const RDSVECTOR &s_tplus1,double dt)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]=(s_tplus1._Value[i]-s_t._Value[i])/dt;
      };

    // Вектор абсолютных значений
    inline const RDSVECTOR Abs(void) const
      { RDSVECTOR v;
        for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          v._Value[i]=fabs(_Value[i]);
        return v;
      };
      
     // Все значения меньше порога
     inline BOOL AllBelow(double value)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          if(_Value[i]>=value)
            return FALSE;
        return TRUE;
      };
 
    // Операторы
    inline const RDSVECTOR operator+(const RDSVECTOR &s)const{return RDSVECTOR(*this,s);};
    inline const RDSVECTOR operator-(const RDSVECTOR &s)const{return RDSVECTOR(*this,s,-1.0);};
    inline const RDSVECTOR &operator+(void)const{return *this;}; // Унарный +
    inline const RDSVECTOR operator-(void)const{return RDSVECTOR(*this,-1.0);}; // Унарный -
    inline const RDSVECTOR operator*(double k)const{return RDSVECTOR(*this,k);};
    friend const RDSVECTOR operator*(double k,const RDSVECTOR &s);
    inline RDSVECTOR &operator+=(const RDSVECTOR &s)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]+=s._Value[i];
        return *this;
      };
    inline RDSVECTOR &operator-=(const RDSVECTOR &s)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]-=s._Value[i];
        return *this;
      };
    inline RDSVECTOR &operator*=(double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]*=k;
        return *this;
      };
    inline RDSVECTOR &operator/=(double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]/=k;
        return *this;
      };
      
      
    // Конструктор по умолчанию
    RDSVECTOR(void)
      {};
    // Конструктор с инициализацией константой
    RDSVECTOR(double c)
      { SetConst(c); };
    // Конструктор суммирования
    RDSVECTOR(const RDSVECTOR &s1,const RDSVECTOR &s2)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]=s1._Value[i]+s2._Value[i];
      };
    RDSVECTOR(const RDSVECTOR &s1,const RDSVECTOR &s2,double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]=s1._Value[i]+k*s2._Value[i];
      };
    // Конструктор умножения на константу
    RDSVECTOR(const RDSVECTOR &s,double v)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _Value[i]=s._Value[i]*v;
      };
};
inline const RDSVECTOR operator*(double k,const RDSVECTOR &s)
  { return RDSVECTOR(s,k); };

#undef RDSVECTOR
#undef RDSVECTORSIZE

#endif
