//------------------------------------------------------------
// Генератор класса для вещественного вектора фиксированной размерности.
//------------------------------------------------------------
// Включение этого файла должно ОБЯЗАТЕЛЬНО предваряться описанием
//   #define RDSVECTOR имя_класса
// При этом генерируется класс с именем "имя_класса".
// Обязательно должно быть описание
//   #define RDSVECTORSIZE размер
// для размера вектора.
// При наличии описания
//   #define RDSVECTORFLAGS тип
// будет добавлен дополнительный вектор флагов данного типа (того же размера).
// При наличии описания
//   #define RDSVECTORALIASES RDSVECTOR_ALIAS(x,0) RDSVECTOR_ALIAS(y,1) ...
// будут добавлены соответствующие псевдонимы.
// Могут также быть добавлены описания, задающие операции с флагами при операциях с векторами:
//   #define RDSVECTORFL_INIT          инициализация_f_константой
//   #define RDSVECTORFL_ADDSUB(f1,f2) сумма_или_разность_f1_и_f2
//   #define RDSVECTORFL_MULDIV(f)     f_при_умножении_или_делении_на_константу
// Например:
//   #define RDSVECTORFL_INIT          0
//   #define RDSVECTORFL_ADDSUB(f1,f2) (f1||f2)
// При отсутствии какого-либо описания соответствующая операция с флагами не выполняется.
//------------------------------------------------------------

#ifdef RDSVECTOR

#ifndef RDSVECTORSIZE
  #error RDSVECTORSIZE must be defined
#else
  #if RDSVECTORSIZE<=0
    #error Bad RDSVECTORSIZE
  #endif
#endif

#include <math.h>
#include <string.h> // Для memcpy

class RDSVECTOR
{ protected:
    double _v[RDSVECTORSIZE];
    #ifdef RDSVECTORFLAGS
      RDSVECTORFLAGS _f[RDSVECTORSIZE];
    #endif

  public:
  
    inline unsigned int Size(void)const{return RDSVECTORSIZE; };
    inline double *Values(void){return _v; };
    inline const double *CValues(void)const{return _v; };
    #ifdef RDSVECTORFLAGS
      inline RDSVECTORFLAGS *Flags(void){return _f; };
      inline const RDSVECTORFLAGS *CFlags(void)const{return _f; };
    #endif


    #ifdef RDSVECTORALIASES
      #ifdef RDSVECTOR_ALIAS
        #undef RDSVECTOR_ALIAS
      #endif
      // Псевдонимы для удобства
      // RDSVECTOR_ALIAS(имя,индекс)
      // Теперь:
      //   x=имя()   - возврат _v[индекс]
      //   _имя()=x  - возврат _v[индекс] как lvalue
      //   имя(x)    - присваивание _v[индекс]
      //   x=имя_f()   - возврат _f[индекс]
      //   _имя_f()=x  - возврат _f[индекс] как lvalue
      //   имя_f(x)    - присваивание _f[индекс]
      #ifdef RDSVECTORFLAGS
        #define RDSVECTOR_ALIAS(var,index) inline double &_##var(void){return _v[index];}; \
             inline double var(void)const{return _v[index];}; \
             inline double var(double __value){return (_v[index]=__value);}; \
             inline RDSVECTORFLAGS &_##var##_f(void){return _f[index];}; \
             inline RDSVECTORFLAGS var##_f(void)const{return _f[index];}; \
             inline RDSVECTORFLAGS var##_f(RDSVECTORFLAGS __value){return (_f[index]=__value);};
      #else
        #define RDSVECTOR_ALIAS(var,index) inline double &_##var(void){return _v[index];}; \
             inline double var(void)const{return _v[index];}; \
             inline double var(double __value){return (_v[index]=__value);};
      #endif
      RDSVECTORALIASES
      #undef RDSVECTOR_ALIAS
      #undef RDSVECTORALIASES
    #endif

    // Максимальное абсолютное значение
    double MaxAbs(void)const 
      { double m=fabs(_v[0]);
        #if RDSVECTORSIZE>1
          for(unsigned int i=1;i<RDSVECTORSIZE;i++)
            { double _m=fabs(_v[i]); if(m<_m) m=_m; }
        #endif
        return m;
      };
      
    // Занести константу
    void SetConst(double c)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _v[i]=c;
      };
    #ifdef RDSVECTORFLAGS
      void SetFlags(RDSVECTORFLAGS f)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            _f[i]=f;
        };
      void SetConstAndFlags(double c,RDSVECTORFLAGS f)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            { _v[i]=c; _f[i]=f; };
        };
    #endif
    
    #ifdef RDSVECTORFLAGS
      // Все ли флаги равны заданному значению
      int AllFlagsAre(RDSVECTORFLAGS f)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            if(_f[i]!=f)
              return 0;
          return 1;
        };
      // Есть ли указанное значение у флага
      int AnyFlagIs(RDSVECTORFLAGS f)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            if(_f[i]==f)
              return 1;
          return 0;
        };
    #endif
      
    // Копировать      
    inline void CopyValuesFrom(const RDSVECTOR &s)
      { memcpy(_v,s._v,RDSVECTORSIZE*sizeof(double)); };
    #ifdef RDSVECTORFLAGS
      inline void CopyFlagsFrom(const RDSVECTOR &s)
        { memcpy(_f,s._f,RDSVECTORSIZE*sizeof(RDSVECTORFLAGS)); };
    #endif
    inline void CopyFrom(const RDSVECTOR &s)
      { CopyValuesFrom(s);
        #ifdef RDSVECTORFLAGS
          CopyFlagsFrom(s);
        #endif
      };
    #ifdef RDSVECTORFLAGS
      // Копировать значения с заданными флагами
      inline void CopyFlaggedValues(const RDSVECTOR &s,RDSVECTORFLAGS f)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            if(s._f[i]==f)
              _v[i]=s._v[i];
        };
    #endif

    // Численная производная по двум точкам
    void CalcDeriv(const RDSVECTOR s_t,const RDSVECTOR &s_tplus1,double dt)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _v[i]=(s_tplus1._v[i]-s_t._v[i])/dt;
      };

    // Вектор абсолютных значений
    inline const RDSVECTOR Abs(void) const
      { RDSVECTOR v;
        for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          v._v[i]=fabs(_v[i]);
        return v;
      };
 
    // Присвоить переменным с указанными значениями флагов указанные значения
    #ifdef RDSVECTORFLAGS
      void SetValueForFlag(RDSVECTORFLAGS f,double c)
        { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
            if(_f[i]==f) 
              _v[i]=c;
        };
    #endif

    // Оператор присваивание
    inline RDSVECTOR &operator=(const RDSVECTOR &s){CopyFrom(s); return *this;};

    // Операторы (не трогают флаги)
    inline const RDSVECTOR operator+(const RDSVECTOR &s)const{return RDSVECTOR(*this,s);};
    inline const RDSVECTOR operator-(const RDSVECTOR &s)const{return RDSVECTOR(*this,s,-1.0);};
    inline const RDSVECTOR &operator+(void)const{return *this;}; // Унарный +
    inline const RDSVECTOR operator-(void)const{return RDSVECTOR(*this,-1.0);}; // Унарный -
    inline const RDSVECTOR operator*(double k)const{return RDSVECTOR(*this,k);};
    friend const RDSVECTOR operator*(double k,const RDSVECTOR &s);
    inline RDSVECTOR &operator+=(const RDSVECTOR &s)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]+=s._v[i];
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_ADDSUB)
              _f[i]=RDSVECTORFL_ADDSUB(_f[i],s._f[i]);
            #endif
          }
        return *this;
      };
    inline RDSVECTOR &operator-=(const RDSVECTOR &s)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]-=s._v[i];
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_ADDSUB)
              _f[i]=RDSVECTORFL_ADDSUB(_f[i],s._f[i]);
            #endif
          }
        return *this;
      };
    inline RDSVECTOR &operator*=(double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]*=k;
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_MULDIV)
              _f[i]=RDSVECTORFL_MULDIV(_f[i]);
            #endif
          }
        return *this;
      };
    inline RDSVECTOR &operator/=(double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]/=k;
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_MULDIV)
              _f[i]=RDSVECTORFL_MULDIV(_f[i]);
            #endif
          }
        return *this;
      };

    RDSVECTOR(void)
      { 
        #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_INIT)
        for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          _f[i]=RDSVECTORFL_INIT;
        #endif
      };
    RDSVECTOR(const RDSVECTOR &s)
      { CopyFrom(s); };
    // Конструктор суммирования
    RDSVECTOR(const RDSVECTOR &s1,const RDSVECTOR &s2)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]=s1._v[i]+s2._v[i];
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_ADDSUB)
              _f[i]=RDSVECTORFL_ADDSUB(s1._f[i],s2._f[i]);
            #endif
          }
      };
    RDSVECTOR(const RDSVECTOR &s1,const RDSVECTOR &s2,double k)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { 
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_ADDSUB)
              RDSVECTORFLAGS f; 
              #ifdef RDSVECTORFL_MULDIV
                f=RDSVECTORFL_MULDIV(s2._f[i]);
              #else
                f=s2._f[i];
              #endif
              _f[i]=RDSVECTORFL_ADDSUB(s1._f[i],f);
            #endif
            _v[i]=s1._v[i]+k*s2._v[i];
          }
      };
    // Конструктор умножения
    RDSVECTOR(const RDSVECTOR &s,double v)
      { for(unsigned int i=0;i<RDSVECTORSIZE;i++)
          { _v[i]=s._v[i]*v;
            #if defined(RDSVECTORFLAGS) && defined(RDSVECTORFL_MULDIV)
              _f[i]=RDSVECTORFL_MULDIV(s._f[i]);
            #endif
          }
      };
};
inline const RDSVECTOR operator*(double k,const RDSVECTOR &s)
  { return RDSVECTOR(s,k); };

#ifdef RDSVECTORFLAGS
  #undef RDSVECTORFLAGS
#endif

#undef RDSVECTOR
#undef RDSVECTORSIZE

#endif
