#ifdef RDSBCPPMULTI_NOAC
  #include <math.h>
#endif
// Многомерная таблица с кусочно-линейной интерполяцией для RDS

// Константы для результата поиска делением пополам
#define RDSBCPPMTSRES_BADARR   0       // Нет массива или в массиве единственный элемент
#define RDSBCPPMTSRES_LEFT     1       // Меньше минимального
#define RDSBCPPMTSRES_RIGHT    2       // Больше максимального
#define RDSBCPPMTSRES_MID      3       // Между двумя элементами
//---------------------------------------------------------------------------

// Индекс таблицы
class rdsbcppMultiTableIndex
{ public:
    double *ExtScale;   // Отсчеты шкалы (указатель на внешние данные)
    int ScaleCount;     // Число отсчетов шкалы

    int Shift;          // Сдвиг в общем массиве при приращении этого индекса

    int Index;          // Установленный снаружи индекс (просто запоминается:
                        // сначала устанавливаются все индексы, потом выбирается элемент
                        // из общего массива функцией GetValuesByIndices)

    double Argument;    // Аргумент для интерполяции по этому индексу
    int InterLeft,InterRight; // Индексы двух отсчетов для интерполяции аргумента
    int InterSearchResult; // Результат поиска соседних отсчетов

    // Данные для вспомогательной интерполяционной таблицы
    int InterShift;     // Сдвиг в массиве таблицы при приращении этого индекса
    int InterIndex;     // Установленный снаружи индекс (два возможных значения: 0 и 1,
                        // просто запоминается, сначала устанавливаются все индексы,
                        // потом выбирается элемент функцией GetInterValuesByIndices)

    // Очистить все
    void Clear(void)
      { ExtScale=NULL; ScaleCount=Shift=0; InterSearchResult=RDSBCPPMTSRES_BADARR; };

    // Найти соседние с аргументом отсчеты для интерполяции
    BOOL SearchArg(void);
    // Найти ближайший отсчет
    int FindClosestArg(double arg);

    // Конструктор
    rdsbcppMultiTableIndex(void)
      { ExtScale=NULL; ScaleCount=Shift=InterShift=0; InterSearchResult=RDSBCPPMTSRES_BADARR; };
};
//---------------------------------------------------------------------------

// Собственно таблица
class rdsbcppMultiTable
{ private:
    void *RdsMatrPtr;           // Переменная RDS
    void *LastDataPtr;          // Указатель на массив данных матрицы в момент последнего подключения (для проверки)

    int Dim;                    // Размерность (число аргументов)
    rdsbcppMultiTableIndex *Index;// Массив шкал

    double *ExtData;            // Данные (указатель на внешние)
    int DataSize;               // Общий размер данных (для справки и поиска минимального/максимального)

    // Временное хранилище размерностей при создании новой таблицы
    int NewDim;
    int *NewScaleCounts;

    // Вспомогательные данные для интерполяции
    // (таблица на один индекс меньше и по каждому индексу всего два отсчета)
    double *InterData;  // Общий массив
    int InterDataSize;  // Размер массива (для справки)

    int LastError;              // Код последней ошибки
      #define RDSBCPPMTERROR_OK                0       // OK
      #define RDSBCPPMTERROR_BADDIM            1       // Не указана или неверная размерность таблицы
      #define RDSBCPPMTERROR_ALREADYSET        2       // Данные уже установлены
      #define RDSBCPPMTERROR_BADARG            3       // Плохой параметр (отрицательное число элементов и т.п.)
      #define RDSBCPPMTERROR_ZEROSCALE         4       // По одной из шкал нет отсчетов
      #define RDSBCPPMTERROR_BADINDEX          5       // Индекс за пределами
      #define RDSBCPPMTERROR_NODATA            6       // Нет массива данных
      #define RDSBCPPMTERROR_SEARCHERR         7       // Ошибка поиска аргумента
      //#define RDSBCPPMTERROR_LOADERR           8       // Ошибка загрузки
      #define RDSBCPPMTERROR_INTERNAL          9       // Непонятная внутренняя ошибка
      #define RDSBCPPMTERROR_BADRDSVAR         10      // Переменная RDS неверного типа
      #define RDSBCPPMTERROR_BADMATR           11      // Матрица RDS содержит неверные данные
    // Установить ошибку
    void SetError(int e){ LastError=e; };

    // Сбросить на ноль индексы во всех размерностях
    void ResetIndices(void);
    // Сбросить на ноль интерполяционые индексы во всех размерностях
    void ResetInterIndices(void);
    // Вычислить следующий интерполяционный индекс (с переносом по индексам)
    BOOL NextInterIndex(void);

    // Индекс без проверок
    rdsbcppMultiTableIndex *GetIndexFAST(int i){return &(Index[i]);};

    // Получить указатель на значение по установленным индексам (может вернуть NULL)
    double *GetValuePtrByIndicesFAST(void);

    //---------- Выборка элемента из интерполяционной таблицы ----------
    // Установить индекс по выбранной размерности
    void SetDimensionInterIndexFAST(int dim,int index){Index[dim].InterIndex=index;};
    // Получить установленный индекс по выбранной размерности
    int GetDimensionInterIndexFAST(int dim){return Index[dim].InterIndex;};
    // Получить во вспомогательной таблице указатель на значение
    // по установленным индексам (нулевой не участвует)
    double * GetInterValuesByIndices(void);
    //------------------------------------------------------------------

    // Настроить на переменную RDS
    BOOL SetupByRDSVar(void);
    // Проверить неизменность подключенной матрицы и переустановить, если она изменилась
    BOOL CheckAndResetup(void);
  public:
    #ifdef RDSBCPPMULTI_NOAC
      // Подключить к указателю на матрицу RDS
      BOOL Connect(void *matrptr){RdsMatrPtr=matrptr; return SetupByRDSVar();};
    #else
      // Подключить к переменной RDS
      BOOL Connect(rdsbcppVarAncestor &var){RdsMatrPtr=var.GetVoidPtr(); return SetupByRDSVar();};
    #endif

    // Получить ошибку
    int GetLastError(void){return LastError;};
    // Сбросить ошибку
    void ResetError(void){LastError=RDSBCPPMTERROR_OK;};

    // Пустая?
    BOOL IsEmpty(void){return ExtData==NULL || Dim==0;};

    // Получить размерность
    int GetDimension(void){return Dim;};
    // Установить размерность
    BOOL SetDimension(int dim);

    // Получить число отсчетов в размерности
    int GetDimensionSize(int dim) // GetIndexSize
      { return (dim>=0 && dim<Dim)?Index[dim].ScaleCount:0; };
    // Установить число отсчетов в размерности
    BOOL SetDimensionSize(int dim,int count);
    // Получить минимальный и максимальный отсчет в размерности
    BOOL GetDimensionRange(int dim,double *pvmin,double *pvmax);

    // Найти максимальное и минимальное значение в таблице
    BOOL GetValueRange(double *pvmin,double *pvmax);

    //---------- Выборка отсчетов размерностей ----------
    // Получить указатель на отсчет размерности
    double *GetDimensionArgPtr(int dim,int index);
    // Получить отсчет размерности
    double GetDimensionArg(int dim,int index,double defvalue=0.0);
    // Установить отсчет размерности
    void SetDimensionArg(int dim,int index,double value);
    // Получить указатель на весь массив отсчетов размерности
    double *GetDimensionArgArray(int dim){return GetDimensionArgPtr(dim,0);};
    //---------------------------------------------------

    //---------- Выборка элемента по индексу ----------
    // Установить индекс по выбранной размерности
    BOOL SetDimensionIndex(int dim,int index);
    // Получить установленный индекс
    int GetDimensionIndex(int dim){return (dim>=0 && dim<Dim)?Index[dim].Index:-1;};
    // Получить указатель на значение по установленным индексам (может вернуть NULL)
    double *GetValuePtrByIndices(void){CheckAndResetup();return GetValuePtrByIndicesFAST();};
    // Получить значение по установленным индексам
    double GetValue(double defvalue=0.0);
    // Записать значение по установленным индексам
    void SetValue(double v);
    //-------------------------------------------------

    //---------- Интерполяция ------------
    // Сброс перед первой интерполяцией
    void ResetInterpolation(void);
    // Установка аргументов для интерполяции
    void SetArgument(int dim,double arg);
    // Получить интерполированное значение (аргументы уже установлены)
    double Interpolate(double defvalue=0.0);
    // Операторы вызова
    double operator()(double x1)
      { SetArgument(0,x1); return Interpolate(); };
    double operator()(double x1,double x2)
      { SetArgument(0,x1); SetArgument(1,x2); return Interpolate(); };
    double operator()(double x1,double x2,double x3)
      { SetArgument(0,x1);
        SetArgument(1,x2);
        SetArgument(2,x3);
        return Interpolate(); };
    double operator()(double x1,double x2,double x3,double x4)
      { SetArgument(0,x1);
        SetArgument(1,x2);
        SetArgument(2,x3);
        SetArgument(3,x4);
        return Interpolate(); };
    double operator()(double x1,double x2,double x3,double x4,double x5)
      { SetArgument(0,x1);
        SetArgument(1,x2);
        SetArgument(2,x3);
        SetArgument(3,x4);
        SetArgument(4,x5);
        return Interpolate(); };
    double operator()(double x1,double x2,double x3,double x4,double x5,double x6)
      { SetArgument(0,x1);
        SetArgument(1,x2);
        SetArgument(2,x3);
        SetArgument(3,x4);
        SetArgument(4,x5);
        SetArgument(5,x6);
        return Interpolate(); };
    //------------------------------------

    // Очистить
    void Clear(void);
    // Размерности установлены - создать таблицу
    BOOL CreateTable(void);
    // Заполнить данные значением
    void FillData(double val);

    // Поиск в массиве делением пополам
    static void SearchInArray(double *arr,int count,double x,
                              int &ileft,int &iright,int &result);
    // Линейная интерполяция
    static double LinInterpol(double x1,double y1,double x2,double y2,double x);

    // Конструктор
    rdsbcppMultiTable(void);
    // Дестркутор
    ~rdsbcppMultiTable(){Clear();};
};
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
// Индекс таблицы - функции
//---------------------------------------------------------------------------
// Найти соседние с аргументом отсчеты для интерполяции
BOOL rdsbcppMultiTableIndex::SearchArg(void)
{
  rdsbcppMultiTable::SearchInArray(ExtScale,ScaleCount,Argument,
                     InterLeft,InterRight,InterSearchResult);
  return InterSearchResult!=RDSBCPPMTSRES_BADARR;
}
//---------------------------------------------------------------------------

// Найти ближайший отсчет
int rdsbcppMultiTableIndex::FindClosestArg(double arg)
{ int left=0,right=ScaleCount-1,result=RDSBCPPMTSRES_BADARR;
  double rleft,rright;
  rdsbcppMultiTable::SearchInArray(ExtScale,ScaleCount,arg,
                             left,right,result);
  if(result==RDSBCPPMTSRES_BADARR)
    return ScaleCount?0:-1;
  rleft=fabs(ExtScale[left]-arg);
  rright=fabs(ExtScale[right]-arg);
  return (rleft<rright)?left:right;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Собственно таблица - функции
//---------------------------------------------------------------------------
// Сбросить на ноль индексы во всех размерностях
void rdsbcppMultiTable::ResetIndices(void)
{
  for(int d=0;d<Dim;d++)
    Index[d].Index=0;
}
//---------------------------------------------------------------------------

// Сбросить на ноль интерполяционые индексы во всех размерностях
void rdsbcppMultiTable::ResetInterIndices(void)
{
  for(int d=0;d<Dim;d++)
    Index[d].InterIndex=0;
}
//---------------------------------------------------------------------------

// Вычислить следующий интерполяционный индекс (с переносом по индексам)
// Возврат - перебор не закончен
BOOL rdsbcppMultiTable::NextInterIndex(void)
{
  // Следующий индекс
  for(int d=Dim-1;d>0;d--)
    { int index=Index[d].InterIndex+1;
      if(index<2)
        { Index[d].InterIndex=index;
          return TRUE;
        }
      // Превысили по индексу d - сбрасываем на ноль и увеличиваем следующий
      Index[d].InterIndex=0;
      if(d==1) // Превышен старший (нулевой при интерполяции не используется, считается отдельно)
        { Index[0].InterIndex=1; // Дополнительный признак
          return FALSE;
        }
    }
  return TRUE;
}
//---------------------------------------------------------------------------

// Получить минимальный и максимальный отсчет в размерности
BOOL rdsbcppMultiTable::GetDimensionRange(int dim,double *pvmin,double *pvmax)
{ // GetIndexRange
  rdsbcppMultiTableIndex *ind;
  CheckAndResetup();
  if(pvmin) *pvmin=0.0;
  if(pvmax) *pvmax=0.0;
  if(dim<0 || dim>=Dim || Index==NULL)
    { SetError(RDSBCPPMTERROR_BADDIM);
      return FALSE;
    }
  ind=Index+dim;
  if(ind->ScaleCount==0 || ind->ExtScale==NULL)
    { SetError(RDSBCPPMTERROR_ZEROSCALE);
      return FALSE;
    }
  if(pvmin) *pvmin=ind->ExtScale[0];
  if(pvmax) *pvmax=ind->ExtScale[ind->ScaleCount-1];
  ResetError();
  return TRUE;
}
//---------------------------------------------------------------------------

// Найти максимальное и минимальное значение в таблице
BOOL rdsbcppMultiTable::GetValueRange(double *pvmin,double *pvmax)
{ double vmin=0.0,vmax=0.0;

  CheckAndResetup();

  if(ExtData==NULL||DataSize==0)
    { SetError(RDSBCPPMTERROR_NODATA);
      return FALSE;
    }
  // В таблице всего DataSize отсчетов
  for(int i=0;i<DataSize;i++)
    { double *base=ExtData+i;
      if(i)
        { if(vmin>(*base)) vmin=(*base);
          if(vmax<(*base)) vmax=(*base);
        }
      else
        vmin=vmax=*base;
    }
  if(pvmin) *pvmin=vmin;
  if(pvmax) *pvmax=vmax;
  return TRUE;
}
//---------------------------------------------------------------------------

// Установить индекс по выбранной размерности
BOOL rdsbcppMultiTable::SetDimensionIndex(int dim,int index)
{ rdsbcppMultiTableIndex *ind;
  if(!ExtData)
    { SetError(RDSBCPPMTERROR_NODATA);
      return FALSE;
    }
  if(dim<0 || dim>=Dim || Index==NULL)
    { SetError(RDSBCPPMTERROR_BADDIM);
      return FALSE;
    }
  ind=GetIndexFAST(dim);
  if(index<0)
    { SetError(RDSBCPPMTERROR_BADINDEX);
      ind->Index=0;
      return FALSE;
    }
  if(index>=ind->ScaleCount)
    { SetError(RDSBCPPMTERROR_BADINDEX);
      ind->Index=ind->ScaleCount-1;
      return FALSE;
    }
  ind->Index=index;
  ResetError();
  return TRUE;
}
//---------------------------------------------------------------------------

// Получить указатель на значение по установленным индексам
double *rdsbcppMultiTable::GetValuePtrByIndicesFAST(void)
{ int n=0;
  if(!ExtData)
    { SetError(RDSBCPPMTERROR_NODATA);
      return NULL;
    }
  for(int i=Dim-1;i>=0;i--)
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      n+=ind->Index*ind->Shift;
    }
  if(n<0 || n>=DataSize)
    { SetError(RDSBCPPMTERROR_BADINDEX);
      return NULL;
    }
  ResetError();
  return ExtData+n;
}
//---------------------------------------------------------------------------

// Получить значение по установленным индексам
double rdsbcppMultiTable::GetValue(double defvalue)
{ double *ptr=GetValuePtrByIndices();
  return ptr?(*ptr):defvalue;
}
//---------------------------------------------------------------------------

// Записать значение по установленным индексам
void rdsbcppMultiTable::SetValue(double v)
{ double *ptr=GetValuePtrByIndices();
  if(ptr) *ptr=v;
}
//---------------------------------------------------------------------------

// Получить во вспомогательной таблице указатель на значение
// по установленным индексам (нулевой не участвует)
double * rdsbcppMultiTable::GetInterValuesByIndices(void)
{ int n=0;
  if(!ExtData)
    { SetError(RDSBCPPMTERROR_NODATA);
      return NULL;
    }
  for(int i=Dim-1;i>0;i--) // НУЛЕВОЙ ИНДЕКС НЕ УЧАСТВУЕТ
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      n+=ind->InterIndex*ind->InterShift;
    }
  if(n<0 || n>=InterDataSize)
    { SetError(RDSBCPPMTERROR_BADINDEX);
      return NULL;
    }
  ResetError();
  return InterData+n;
}
//---------------------------------------------------------------------------

// Сброс перед первой интерполяцией
void rdsbcppMultiTable::ResetInterpolation(void)
{ for(int i=0;i<Dim;i++)
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      ind->InterSearchResult=RDSBCPPMTSRES_BADARR;
      ind->Argument=0.0;
    }
}
//---------------------------------------------------------------------------

// Установка аргументов для интерполяции
void rdsbcppMultiTable::SetArgument(int dim,double arg)
{ if(dim>=0 && dim<Dim)
    Index[dim].Argument=arg;
}
//---------------------------------------------------------------------------

// Получить интерполированное значение (аргументы уже установлены)
double rdsbcppMultiTable::Interpolate(double defvalue)
{ double x1,x2;

  if(!CheckAndResetup())
    { SetError(RDSBCPPMTERROR_NODATA);
      return defvalue;
    }

  if(!ExtData)
    { SetError(RDSBCPPMTERROR_NODATA);
      return defvalue;
    }

  // Поиск в индексах соседних с аргументом отсчетов
  for(int i=0;i<Dim;i++)
    if(!Index[i].SearchArg())
      { SetError(RDSBCPPMTERROR_SEARCHERR);
        return defvalue;
      }


  // Интерполяция по старшему (нулевому) индексу -
  // данные берутся из основной таблицы Data, заполняется
  // интерполяционная таблица InterData
  // Index[i].InterIndex используется не совсем обычно:
  // 0 - в общей таблице берется InterLeft, 1 - берется InterRight
  ResetInterIndices();
  // Считываем левое и правое значения аргумента нулевого индекса
  x1=Index->ExtScale[Index->InterLeft];
  x2=Index->ExtScale[Index->InterRight];
  // Перебор всех индексов, кроме нулевого
  for(;;)
    { double *valleft,*valright,*interp;
      // Заполняем во всех, кроме нулевого, Index по InterIndex
      for(int d=1;d<Dim;d++)
        { rdsbcppMultiTableIndex *ind=GetIndexFAST(d);
          ind->Index=(ind->InterIndex)?ind->InterRight:ind->InterLeft;
        }
      // Получаем две точки: (0,i1,i2,...) и (1,i1,i2,...)
      Index->Index=Index->InterLeft;
      valleft=GetValuePtrByIndicesFAST();
      Index->Index=Index->InterRight;
      valright=GetValuePtrByIndicesFAST();
      // Находим в интерполяционной таблице точку (i1,i2,...)
      interp=GetInterValuesByIndices();
      // Пишем в интерполяционную таблицу результат интерполяции
      *interp=LinInterpol(x1,*valleft,x2,*valright,Index->Argument);
      // Берем следующий индекс
      if(!NextInterIndex())
        break;
    }

  // Теперь можно интерполировать InterData индекс за индексом
  for(int d=1;d<Dim;d++)
    { // d - индекс, по которому нужно интерполировать
      rdsbcppMultiTableIndex *c_ind=GetIndexFAST(d);
      x1=c_ind->ExtScale[c_ind->InterLeft];
      x2=c_ind->ExtScale[c_ind->InterRight];

      ResetInterIndices();
      for(;;)
        { double *valleft,*valright;
          // Получаем две точки: (...,0,i{d+1},i{d+2},...) и (...,1,i{d+1},i{d+2},...)
          c_ind->InterIndex=0;
          valleft=GetInterValuesByIndices();
          c_ind->InterIndex=1;
          valright=GetInterValuesByIndices();
          // Интерполируем и пишем результат в valleft
          *valleft=LinInterpol(x1,*valleft,x2,*valright,c_ind->Argument);
          c_ind->InterIndex=0;
          NextInterIndex();
          if(c_ind->InterIndex) // Перенос в текущий индекс - все перебрано
            break;
        }

    }

  // После всех циклов в начале InterData получается результат интерполяции
  ResetError();
  return *InterData;
}
//---------------------------------------------------------------------------

// Очистить
void rdsbcppMultiTable::Clear(void)
{
  Dim=DataSize=0;
  if(Index)
    delete[] Index;
  if(InterData)
    delete[] InterData;
  Index=NULL;
  ExtData=InterData=NULL;
  NewDim=0;
  if(NewScaleCounts)
    delete[] NewScaleCounts;
  NewScaleCounts=NULL;
  ResetError();
  LastDataPtr=NULL;
}
//---------------------------------------------------------------------------

// Заполнить данные значением
void rdsbcppMultiTable::FillData(double val)
{
  CheckAndResetup();
  if(ExtData)
    for(int i=0;i<DataSize;i++)
      ExtData[i]=val;
}
//---------------------------------------------------------------------------

// Поиск в массиве делением пополам (static)
// arr    - массив (отсортированный по возрастанию)
// count  - элементов в массиве
// x      - искомое значение (аргумент)
// ileft  - вход: левый индекс прошлого поиска, выход: найденный левый индекс
// iright - вход: правый индекс прошлого поиска, выход: найденный правый индекс
// result - вход: результат прошлого поиска, выход: новый результат
void rdsbcppMultiTable::SearchInArray(double *arr,int count,double x,
                                      int &ileft,int &iright,int &result)
{ double l,r;
  int _ileft,_iright,maxind=count-1;

  if(arr==NULL || count<2)
    { result=RDSBCPPMTSRES_BADARR;
      return;
    }

  // Если прошлый поиск был успешен - проверяем, может, новый не нужен
  switch(result)
    { case RDSBCPPMTSRES_LEFT: // прошлый x был левее всего массива
        if(x<=arr[0]) // ничего не изменилось
          return;
        if(x<arr[1]) // теперь между первым и вторым - индексы те же, результат другой
          { result=RDSBCPPMTSRES_MID;
            return;
          }
        // Необходим новый поиск
        _ileft=0;
        _iright=maxind;
        break;

      case RDSBCPPMTSRES_RIGHT: // прошлый x был правее всего массива
        if(x>=arr[iright]) // ничего не изменилось
          return;
        if(x>=arr[ileft]) // теперь между предпоследним и последним - индексы те же, результат другой
          { result=RDSBCPPMTSRES_MID;
            return;
          }
        // Необходим новый поиск
        _ileft=0;
        _iright=maxind;
        break;

      case RDSBCPPMTSRES_MID: // x был между ileft и iright
        l=arr[ileft];
        r=arr[iright];
        if(x>=l)
          { // все еще правее ileft
            if(x<=r) // и левее iright - нет изменений
              return;
            // уехал куда-то вправо от iright
            if(iright==maxind) // правее ничего нет
              { // поскольку iright==maxind, ileft==maxind-1
                // меняется только результат
                result=RDSBCPPMTSRES_RIGHT;
                return;
              }
            // правее есть еще элемент массива
            if(x<=arr[iright+1])
              { // между iright и следующим
                ileft=iright;
                iright++;
                result=RDSBCPPMTSRES_MID;
                return;
              }
            // уехал далеко вправо - нужен новый поиск
            _ileft=iright+1; // мы уже знаем, что он правее iright+1
            _iright=maxind;
            break;
          } // if(x>=l)
        // x левее ileft
        if(ileft==0) // левее ничего нет - меняется только результат
          { result=RDSBCPPMTSRES_LEFT;
            return;
          }
        // левее есть что-то
        if(x>=arr[ileft-1])
          { // между ileft и его соседом слева
            iright=ileft;
            ileft--;
            result=RDSBCPPMTSRES_MID;
            return;
          }
        // уехал далеко влево - нужен новый поиск
        _ileft=0;
        _iright=ileft+1; // мы уже знаем, что x<arr[ileft-1]
        break;

      default:
        // Необходим новый поиск
        _ileft=0;
        _iright=maxind;
    }


  // Если дошли досюда, значит, нужен поиск
  if(x<=arr[0]) // левее массива
    { ileft=0;
      iright=1;
      result=RDSBCPPMTSRES_LEFT;
      return;
    }
  if(x>=arr[maxind]) // правее массива
    { ileft=maxind-1;
      iright=maxind;
      result=RDSBCPPMTSRES_RIGHT;
      return;
    }
  if(_ileft==_iright) // На всякий случай
    { _ileft=0; _iright=maxind; }

  // x где-то внутри массива между _ileft и _iright
  for(;;)
    { int m;
      if(_iright==_ileft+1) // Сузили до двух соседних
        break; // поиск закончен
      m=(_ileft+_iright)/2; // середина
      if(x<arr[m]) // в левой половине
        _iright=m;
      else // в правой половине
        _ileft=m;
    }
  ileft=_ileft;
  iright=_iright;
  result=RDSBCPPMTSRES_MID;
}
//---------------------------------------------------------------------------

// Линейная интерполяция (static)
double rdsbcppMultiTable::LinInterpol(double x1,double y1,double x2,double y2,double x)
{ double k;
  k=(y2-y1)/(x2-x1);
  return k*x+(y2-k*x2);
}
//---------------------------------------------------------------------------

// Конструктор
rdsbcppMultiTable::rdsbcppMultiTable(void)
{
  Dim=0;
  Index=NULL;
  ExtData=InterData=NULL;
  LastError=RDSBCPPMTERROR_OK;
  RdsMatrPtr=NULL;
  LastDataPtr=NULL;
  NewDim=0;
  NewScaleCounts=NULL;
}
//---------------------------------------------------------------------------

// Проверить неизменность подключенной матрицы и переустановить, если она изменилась
BOOL rdsbcppMultiTable::CheckAndResetup(void)
{ void *array=NULL;
  double *args;
  int *indices;
  int old_dim=0;

  if(!RdsMatrPtr)
    return FALSE;
  // Матрица RDS есть
  if(RDS_ARRAYEXISTS(RdsMatrPtr))
    array=RDS_ARRAYDATA(RdsMatrPtr);
  if(array==LastDataPtr) // Массив не изменился
    return TRUE;

  // Массив изменился
  if(array==NULL)
    { SetupByRDSVar();
      return FALSE;
    }

  // Запоминаем текущие установленные значения индексов и аргументов
  if(Dim)
    { old_dim=Dim;
      args=new double[Dim];
      indices=new int[Dim];
      for(int i=0;i<Dim;i++)
        { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
          args[i]=ind->Argument;
          indices[i]=ind->Index;
        }
    }

  // Пересоединяем
  if(SetupByRDSVar())
    { if(old_dim)
        for(int i=0;i<Dim;i++)
          { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
            if(i<old_dim) break;
              { args[i]=ind->Argument;
                indices[i]=ind->Index;
              }
            // ind->InterSearchResult=RDSBCPPMTSRES_BADARR уже есть в SetupByRDSVar
          }
    }

  if(old_dim)
    { delete[] args;
      delete[] indices;
    }
  return TRUE;
}
//---------------------------------------------------------------------------

// Настроить на переменную RDS
BOOL rdsbcppMultiTable::SetupByRDSVar(void)
{ int n,total_mul,total_sum;
  double *Array,errorvalue;
  rdsGetHugeDouble(&errorvalue);
  Clear();
  if(!RdsMatrPtr)
    { SetError(RDSBCPPMTERROR_BADRDSVAR);
      return FALSE;
    }
  if(rdsGetVarArrayElementType(RdsMatrPtr)!=RDS_VARTYPE_DOUBLE)
    { RdsMatrPtr=NULL;
      SetError(RDSBCPPMTERROR_BADRDSVAR);
      return FALSE;
    }

  if(!RDS_ARRAYEXISTS(RdsMatrPtr)) // Пустая матрица
    return TRUE;

  // Матрица не пустая
  n=RDS_ARRAYCOLS(RdsMatrPtr)*RDS_ARRAYROWS(RdsMatrPtr);
  Array=(double*)(LastDataPtr=RDS_ARRAYDATA(RdsMatrPtr));
  if(n<1)
    { SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }
  if(Array[0]==errorvalue)
    { Clear();
      SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }
  Dim=(int)(Array[0]);
  if(Dim<=0)
    { Clear();
      SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }

  // Дальше должны следовать числа отсчетов по осям - всего Dim штук:
  // Array[1]...Array[Dim]
  if(n<Dim+1)
    { Clear();
      SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }

  // Размерность определена, числа отсчетов есть
  total_mul=1;
  total_sum=0;
  Index=new rdsbcppMultiTableIndex[Dim];
  ResetInterpolation();
  for(int i=0;i<Dim;i++)
    { double d=Array[i+1];
      int m;
      if(d==errorvalue)
        { total_sum=0;
          break;
        }
      m=(int)d;
      Index[i].ScaleCount=m;
      total_mul*=m;
      total_sum+=m;
    }
  if(total_mul<=0 || total_sum<=0) // Что-то не так
    { Clear();
      SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }

  // Дальше должны идти отсчеты шкал. Сначала проверяем размер.
  // Общий размер должен быть:
  // 1 (Dim) + Dim (размеры шкал) + total_sum (отсчеты аргументов) + total_mul (сами данные)
  if(n!=1+Dim+total_sum+total_mul)
    { Clear();
      SetError(RDSBCPPMTERROR_BADMATR);
      return FALSE;
    }
  // Размер правильный - грузим шкалы
  Array+=Dim+1;
  for(int i=0;i<Dim;i++)
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      ind->ExtScale=Array;
      Array+=ind->ScaleCount;
    }

  // Теперь Array указывает на начало блока значений
  ExtData=Array;
  DataSize=total_mul;

  // Перемножение всех размерностей
  n=1;
  for(int i=Dim-1;i>=0;i--)
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      ind->Shift=n;
      n*=ind->ScaleCount;
      ind->InterSearchResult=RDSBCPPMTSRES_BADARR;
    }

  // Вычисление размеров InterData - там нет старшего (нулевого) индекса
  // и всего по два отсчета в оставшихся
  n=1;
  Index[0].InterShift=0; // Старший индекс не участвует
  for(int i=Dim-1;i>0;i--)
    { rdsbcppMultiTableIndex *ind=GetIndexFAST(i);
      ind->InterShift=n;
      n*=2;
    }
  InterData=new double[InterDataSize=n];

  ResetError();
  return TRUE;
}
//---------------------------------------------------------------------------


// Установить размерность
BOOL rdsbcppMultiTable::SetDimension(int dim)
{
  if(dim<=0)
    { SetError(RDSBCPPMTERROR_BADARG);
      return FALSE;
    }
  NewDim=dim;
  if(NewScaleCounts)
    delete[] NewScaleCounts;
  NewScaleCounts=new int[NewDim];
  memset(NewScaleCounts,0,NewDim*sizeof(int));
  ResetError();
  return TRUE;
}
//---------------------------------------------------------------------------

// Установить число отсчетов в размерности
BOOL rdsbcppMultiTable::SetDimensionSize(int dim,int count)
{ // SetIndexSize
  if(dim<0 || dim>=NewDim || count<=0)
    { SetError(RDSBCPPMTERROR_BADARG);
      return FALSE;
    }
  NewScaleCounts[dim]=count;
  ResetError();
  return TRUE;
}
//---------------------------------------------------------------------------

// Размерности установлены - создать таблицу (в переменной RDS)
BOOL rdsbcppMultiTable::CreateTable(void)
{ // AllocateData
  int total,total_mul;
  double *Array;

  // Общий размер должен быть:
  // 1 (Dim) + Dim (размеры шкал) + total_sum (отсчеты аргументов) + total_mul (сами данные)

  if(NewDim==0 || NewScaleCounts==NULL)
    { Clear();
      SetError(RDSBCPPMTERROR_BADARG);
      return FALSE;
    }
  if(RdsMatrPtr==NULL)
    { Clear();
      SetError(RDSBCPPMTERROR_BADRDSVAR);
      return FALSE;
    }

  // Вычисляем общий размер
  total=NewDim+1;
  total_mul=1;
  for(int i=0;i<NewDim;i++)
    { total+=NewScaleCounts[i];
      total_mul*=NewScaleCounts[i];
    }
  total+=total_mul;
  if(total<=0)
    { Clear();
      SetError(RDSBCPPMTERROR_INTERNAL);
      return FALSE;
    }

  if(!rdsResizeVarArray(RdsMatrPtr,total,1,FALSE,NULL))
    { Clear();
      rdsResizeVarArray(RdsMatrPtr,0,0,FALSE,NULL);
      SetError(RDSBCPPMTERROR_INTERNAL);
      return FALSE;
    }
  Array=(double*)RDS_ARRAYDATA(RdsMatrPtr);

  // Заполняем размерности
  Array[0]=NewDim;
  for(int i=0;i<NewDim;i++)
    Array[i+1]=NewScaleCounts[i];

  // Заполняем все остальное по матрице
  return SetupByRDSVar();
}
//---------------------------------------------------------------------------

// Получить указатель на отсчет размерности
double *rdsbcppMultiTable::GetDimensionArgPtr(int dim,int index)
{ rdsbcppMultiTableIndex *ind;
  if(!CheckAndResetup())
    return NULL;
  if(dim<0 || dim>=Dim || Index==NULL)
    { SetError(RDSBCPPMTERROR_BADDIM);
      return NULL;
    }
  ind=GetIndexFAST(dim);
  if(ind->ExtScale==NULL)
    { SetError(RDSBCPPMTERROR_ZEROSCALE);
      return NULL;
    }
  if(index<0 || index>=ind->ScaleCount)
    { SetError(RDSBCPPMTERROR_BADINDEX);
      return NULL;
    }
  ResetError();
  return ind->ExtScale+index;
}
//---------------------------------------------------------------------------

// Получить отсчет размерности
double rdsbcppMultiTable::GetDimensionArg(int dim,int index,double defvalue)
{ double *ptr=GetDimensionArgPtr(dim,index);
  return ptr?(*ptr):defvalue;
}
//---------------------------------------------------------------------------

// Установить отсчет размерности
void rdsbcppMultiTable::SetDimensionArg(int dim,int index,double value)
{ double *ptr=GetDimensionArgPtr(dim,index);
  if(ptr)
    *ptr=value;
}
//---------------------------------------------------------------------------

